function [ImageStack, InfoFile] = ReadZStack_MultiChannel_Optimized(FileName, NumImage, ImageSize, StepInterval, TotalNumChannels, ChannelIndex, NumberOfFramesPerZ, FirstCenterFrameIndex)
% Updated on 2023.02.01 by Jonathan S.D. Radda for optimization:
% 1. Sequentially loading only needed frames of raw image lowers loading time and memory overhead
% 2. Preallocation of ImageStack variable

%% New variables
% NumberOfFramesPerZ: Number of frames to average for each z-step in the final image
% FirstCenterFrameIndex: Number of frame in first set of frames of first color channel over which NumberOfFramesPerZ will be centered

%% Old variables
% FileName: Name of image file
% NumImage: Total number of frames in image
% ImageSize: Size of image in x or y dimension
% StepInterval: Number of frames per z-height per color
% TotalNumChannels: Number of color channels used in image (1 for single-color image)
% ChannelIndex: Index of color channel to load from image (1 for single-color image)

%% Choosing frames.
% This section is separated for readability and customization purposes
numZHeights = floor(NumImage/(StepInterval*TotalNumChannels));
frameRange = floor(NumberOfFramesPerZ/2);
startFrame = zeros(1,numZHeights);
endFrame = zeros(1,numZHeights);
for i = 1:numZHeights
    %% ODD NUMBER of FRAMES
    % This will average an odd number of frames
    startFrame(i) = ((i-1)*TotalNumChannels+ChannelIndex-1)*StepInterval+FirstCenterFrameIndex-frameRange;
    endFrame(i) = startFrame(i) + NumberOfFramesPerZ - 1;
    
    %% EVEN NUMBER of FRAMES
    % This code may be used if an even number of frames per z are to be
    % averaged. You will need to change FirstCenterFrameIndex to
    % FirstFrameIndex, which will represent the first frame to be averaged
    % in the first color channel in the first z-plane.
%     startFrame(i) = ((i-1)*TotalNumChannels+ChannelIndex-1)*StepInterval+FirstFrameIndex;
%     endFrame(i) = startFrame(i) + NumberOfFramesPerZ - 1;
end

%% Loading image
% using parfor actually slows this section down in my tests (loading 3 color 1536x1536x751 z-stack)
ImageStack = zeros(ImageSize,ImageSize,numZHeights);
for i = 1:numZHeights
    [MovieFP, InfoFile] = ReadDax([FileName, '.dax'],'startFrame', startFrame(i), 'endFrame', endFrame(i), 'verbose', false);
    ImageStack(:,:,i) = mean(MovieFP,3);
end